/*
 * Authentication system for Lachesis
 */

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include "auth.h"
#include "loaddata.h"
#include "ircint.h"
#include "utils.h"
#include "cmd/root.h"

Auth_PUser   Auth_TUser  ::master = NULL;
Auth_PUser   Auth_TUser  ::first  = NULL;
Auth_PUser   Auth_TUser  ::last   = NULL;
uint16       Auth_TUser  ::length = 0;
Auth_PTicket Auth_TTicket::first  = NULL;
Auth_PTicket Auth_TTicket::last   = NULL;
Auth_PTicket Auth_TTicket::stale  = NULL;
Auth_PUHCB   Auth_TUHCB  ::first  = NULL;
Auth_PUHCB   Auth_TUHCB  ::last   = NULL;

Auth_TUser::Auth_TUser(const char *s_acct, const char *s_pass, TF s_master)
{
#ifdef DEBUG
  fputs("DEBUG - Auth_TUser ctor called\n", stderr);
#endif
  if (last!=NULL) {
    prev = last;
    prev->next = this;
    next = NULL;
    last = this;
    length++;
  } else {
    first = last = this;
    prev = next = NULL;
    length = 1;
  }
  chans = NULL;
  dcc_cookie = NULL;
  dcc_active = FALSE;
  security = TRUE;
  usemsg = FALSE;
#ifndef RPGSERV
  sounds = TRUE;
  debug = FALSE;
#endif
  Util_Copy(acct, s_acct, ALLOC_ACCT);
  Util_Copy(pass, s_pass, ALLOC_PASS);
  *nick = 0;
  *user = 0;
  *host = 0;
  if (s_master) {
    if (master!=NULL) {
      fputs("Error: Multiple master accounts. Exiting.", stderr);
      exit(255);
    }
    master = this;
  }
  if (GetUser(acct)!=this) {
    fprintf(stderr, "Multiple account instances for %s\n", acct);
    exit(255);
  }
}

Auth_TTicket::Auth_TTicket(const char *s_nick, const char *s_user, 
			   const char *s_host, const char *channel, 
			   IRC_PCL s_chans)
{
#ifdef DEBUG
  fputs("DEBUG - Auth_TTicket ctor called\n", stderr);
#endif
  if (last!=NULL) {
    prev = last;
    prev->next = this;
    next = NULL;
    last = this;
  } else {
    first = last = this;
    prev = next = NULL;
  }
  if (channel==NULL)
    chans = s_chans;
  else {
    chans = new IRC_TCL(NULL);
    Util_Copy(chans->chan, channel, ALLOC_CHAN);
  }
  Util_Copy(nick, s_nick, ALLOC_NICK);
  Util_Copy(user, s_user, ALLOC_USER);
  Util_Copy(host, s_host, ALLOC_HOST);
}

void Auth_TUser::Delete(Auth_PUser user)
{
#ifdef DEBUG
  fputs("DEBUG - Auth_TUser::Delete called\n", stderr);
#endif
  if (user->prev!=NULL)
    user->prev->next = user->next;
  if (user->next!=NULL)
    user->next->prev = user->prev;
  length--;
  if (first==user)
    first = user->next;
  if (last==user)
    last = user->prev;
  if (user->chans!=NULL) {
#ifdef DEBUG
    fputs("DEBUG - Flushing non-empty chans pointer.\n", stderr);
#endif
    IRC_TCL::Flush(user->chans);
  }
  if (user->dcc_cookie!=NULL) {
    if (user->dcc_active) {
#ifdef DEBUG
      fputs("DEBUG - Closing non-empty DCC cookie.\n", stderr);
#endif
      IRC_DCCChatClose(user->dcc_cookie);
    } else {
#ifdef DEBUG
      fputs("DEBUG - Rejecting non-empty DCC cookie.\n", stderr);
#endif
      IRC_DCCChatReject(user->dcc_cookie);
    }
  }
  if (master == user)
    master = NULL;
  delete [] user;
}

void Auth_TTicket::Delete(Auth_PTicket ticket)
{
#ifdef DEBUG
  fputs("DEBUG - Auth_TTicket::Delete called\n", stderr);
#endif
  if (ticket->prev!=NULL)
    ticket->prev->next = ticket->next;
  if (ticket->next!=NULL)
    ticket->next->prev = ticket->prev;
  if (first==ticket)
    first = ticket->next;
  if (last==ticket)
    last = ticket->prev;
  delete [] ticket;
}

void Auth_TTicket::CleanStale()
{
  if (stale!=NULL) {
    Delete(stale);
    stale = NULL;
  }
}

void Auth_TTicket::DeleteMe(Auth_PTicket s_stale)
{
  // Prepare a stale ticket for deletion
  CleanStale();
  stale = s_stale;
}

void Auth_TUser::FlushList()
{
  while (first!=NULL)
    Delete(first);
}

void Auth_TTicket::FlushList()
{
  stale = NULL;
  while (first!=NULL) {
    if (first->chans!=NULL)
      IRC_TCL::Flush(first->chans);  // Flushing is a special case where we
    Delete(first);                   // take responsibility for chans.
  }
}

Auth_PUser Auth_TUser::GetUser(const char *s_acct)
{
  if (s_acct==NULL)
    return first;
  Auth_PUser temp = first;
  while (temp!=NULL) {
    if (strcasecmp(temp->acct, s_acct)==0)
      return temp;
    temp = temp->GetNext();
  }
  return NULL;
}

Auth_PUser Auth_TUser::GetMaster()
{
  return master;
}

const char * Auth_TUser::GetAcct() const
{
  return acct;
}

const char * Auth_TUser::GetNick() const
{
  return nick;
}

const char * Auth_TTicket::GetNick() const
{
  return nick;
}

const char * Auth_TUser::GetUser() const
{
  return user;
}

const char * Auth_TTicket::GetUser() const
{
  return user;
}

const char * Auth_TUser::GetHost() const
{
  return host;
}

const char * Auth_TTicket::GetHost() const
{
  return host;
}

IRC_PCL Auth_TUser::GetChans()
{
  return chans;
}

IRC_PCL Auth_TTicket::GetChans()
{
  CleanStale();
  return chans;
}

TF Auth_TUser::Login(const char *s_pass, const char *s_nick, 
		     const char *s_user, const char *s_host, IRC_PCL s_chans)
{
#ifdef DEBUG
  fputs("DEBUG - Auth_TUser::Login called\n", stderr);
#endif
  if (strcmp(pass, s_pass)!=0)
    return FALSE;
  if (s_chans==NULL)  // We're not tracking this user
    if (security)
      return FALSE;  // Can't log in if we're not tracking!
  Util_Copy(nick, s_nick, ALLOC_NICK);
  Util_Copy(user, s_user, ALLOC_USER);
  Util_Copy(host, s_host, ALLOC_HOST);
  chans = s_chans;
  return TRUE;
}

void Auth_TUser::Logout()
{
#ifdef DEBUG
  fputs("DEBUG - Auth_TUser::Logout called\n", stderr);
#endif
  *nick = 0;
  *user = 0;
  *host = 0;
  chans = NULL;
  if (dcc_cookie!=NULL) {
    if (dcc_active)
      IRC_DCCChatClose(dcc_cookie);
    else
      IRC_DCCChatReject(dcc_cookie);
    dcc_cookie=NULL;
    dcc_active=FALSE;
  }
}

void Auth_TUser::AddTracking(const char *channel)
{
  if (chans==NULL) {
    fprintf(stderr, 
	  "Error: Added a channel to tracking, but we're not tracking!\n\
Channel: %s  Account: %s  Mask: %s@%s\n", channel, acct, user, host);
    return ;
  }
  IRC_PCL temp;
  for (temp=chans;temp!=NULL;temp=temp->GetNext())
    if (strcasecmp(temp->chan, channel)==0)
      return ;
  chans = new IRC_TCL(chans);
  Util_Copy(chans->chan, channel, ALLOC_CHAN);
}

void Auth_TTicket::AddTracking(const char *channel)
{
  CleanStale();
  IRC_PCL temp;
  for (temp=chans;temp!=NULL;temp=temp->GetNext())
    if (strcasecmp(temp->chan, channel)==0)
      return ;
  chans = new IRC_TCL(chans);
  Util_Copy(chans->chan, channel, ALLOC_CHAN);
}

void Auth_TUser::RemTracking(const char *channel)
{
  if (chans==NULL)
    return ;
  IRC_PCL temp = chans;
  while (temp!=NULL) {
    if (strcasecmp(temp->chan, channel)==0) {
      if (temp == chans)
	chans = chans->GetNext();
      IRC_TCL::Delete(temp);
      return ;
    }
    temp = temp->GetNext();
  }
}

void Auth_TTicket::RemTracking(const char *channel)
{
  CleanStale();
  IRC_PCL temp = chans;
  while (temp!=NULL) {
    if (strcasecmp(chans->chan, channel)==0) {
      if (temp == chans)
	chans = chans->GetNext();
      IRC_TCL::Delete(temp);
      if (chans == NULL)
	DeleteMe(this);
      return ;
    }
    temp = temp->GetNext();
  }
}

void Auth_TUser::ChangeTracking(const char *s_nick)
{
  Util_Copy(nick, s_nick, ALLOC_NICK);
}

void Auth_TTicket::ChangeTracking(const char *s_nick)
{
  CleanStale();
  Util_Copy(nick, s_nick, ALLOC_NICK);
}

void Auth_TUser::StartTracking(const char *channel, const char *s_nick)
{
  if (chans!=NULL) {
    fputs("Error: Old tracking list still present; deleting.\n", stderr);
    IRC_TCL::Flush(chans);
  }
  chans = new IRC_TCL(NULL);
  Util_Copy(chans->chan, channel, ALLOC_CHAN);
  Util_Copy(nick, s_nick, ALLOC_NICK);
}

void Auth_TUser::FlushTracking()
{
  if (chans==NULL)
    return ;
  IRC_TCL::Flush(chans);
  chans = NULL;
  if (security)
    *nick = 0;
}

void Auth_TTicket::FlushTracking()
{
  IRC_TCL::Flush(chans);
  chans = NULL;
  DeleteMe(this);
}

void Auth_TUser::FlushAllTracking()
{
  Auth_PUser temp;
  temp = first;
  while (temp!=NULL) {
    if (temp->chans!=NULL)
      temp->FlushTracking();
    temp = temp->GetNext();
  }
}

TF Auth_TUser::IsTracking(const char *channel) const
{
  if (channel==NULL)
    return chans!=NULL;
  IRC_PCL temp;
  temp = chans;
  while (temp!=NULL) {
    if (strcasecmp(temp->chan, channel)==0)
      return TRUE;
    temp = temp->GetNext();
  }
  return FALSE;
}

TF Auth_TTicket::IsTracking(const char *channel) const
{
  if (stale==this)
    return FALSE;
  IRC_PCL temp;
  temp = chans;
  while (temp!=NULL) {
    if (strcasecmp(temp->chan, channel)==0)
      return TRUE;
    temp = temp->GetNext();
  }
  return FALSE;
}

void * Auth_TUser::RegisterDCC(void *cookie)
{
  void *temp;
  temp = dcc_cookie;
  dcc_cookie = cookie;
  return temp;
}

void Auth_TUser::SetDCCStatus(TF flag)
{
  if (flag) {
    if (dcc_cookie==NULL)
      return ;
    dcc_active = TRUE;
  } else {
    if (dcc_cookie!=NULL) {
      if (dcc_active)
	IRC_DCCChatClose(dcc_cookie);
      else
	IRC_DCCChatReject(dcc_cookie);
      dcc_cookie = NULL;
    }
    dcc_active = FALSE;
  }
}

TF Auth_TUser::GetDCCStatus() const
{
  if (dcc_cookie!=NULL)
    return dcc_active;
  return FALSE;
}

void Auth_TUser::SendMessage(const char *message)
{
  if (dcc_active&&dcc_cookie!=NULL) {
    IRC_DCCChatMessage(dcc_cookie, message);
    return ;
  }
  if (*nick) {
    if (chans!=NULL) {
      if (usemsg)
	IRC_MSG(nick, message);
      else
	IRC_Notice(nick, message);
      return ;
    }
    if (!security&&*user&&*host) {
      Auth_PUHCB temp = new Auth_TUHCB(&Auth_VerifyAndSendMSG, nick);
      temp->data = malloc(ALLOC_BUFFER);
      Util_Copy((char *)(temp->data), message, ALLOC_BUFFER);
      IRC_Userhost(nick);
    }
  }
}

void Auth_TTicket::SendMessage(const char *message)
{
  CleanStale();
  IRC_Notice(nick, message);
}

void Auth_VerifyAndSendMSG(Auth_PUHCB uhcb)
{
  Auth_PUser temp = Auth_TUser::MatchNick(uhcb->GetNick());
  if (temp!=NULL) {
    if (strcmp(temp->user, uhcb->GetUser())==0 &&
	strcmp(temp->host, uhcb->GetHost())==0) {
      if (temp->usemsg)
	IRC_MSG(uhcb->GetNick(), (char *)(uhcb->data));
      else
	IRC_Notice(uhcb->GetNick(), (char *)(uhcb->data));
    }
    else
      *(temp->nick) = 0;  // Invalid nick.
  }
  free (uhcb->data);
}

TF Auth_Add(void **tref, const char *acct, const char *pass)
{
  Auth_PUser temp;
  Auth_PTicket ticket;
  IRC_PCL chans;
  ticket = (Auth_PTicket)(*tref);
  temp = Auth_TUser::GetUser(acct);
  if (temp==NULL)
    return FALSE;

  chans = ticket->GetChans();
  if (strcmp(chans->chan, "boguschan")==0)
    chans = NULL;
  if (temp->Login(pass, ticket->GetNick(), ticket->GetUser(), 
		  ticket->GetHost(), chans)) {
    *tref = temp; // We need to pass back a valid user ticket.
    if (chans==NULL) {
      chans = ticket->GetChans();
      delete [] chans;
    }
    Auth_TTicket::Delete(ticket);
    return TRUE;
  }
  return FALSE;
}

void Auth_Track(const char *nick, const char *channel, TF flag)
{
  if (strcasecmp(nick, IRC_GetNick())==0)
    return ;
  if (flag) {
    Auth_PUser tuser;
    Auth_PTicket ttkt;
    if ((tuser = Auth_TUser::MatchNick(nick))!=NULL)
      if (tuser->IsTracking(NULL)) {  // We are already tracking this user
	if (tuser->IsTracking(channel))
	  return ;
	tuser->AddTracking(channel);
	return ;
      }
    if ((ttkt = Auth_TTicket::MatchNick(nick))!=NULL)
      if (!ttkt->IsTracking(channel)) {
	ttkt->AddTracking(channel);
	return ;
      }
    Auth_PUHCB temp = new Auth_TUHCB(&Auth_TrackCR, nick);
    temp->data = (void *)(new char[ALLOC_CHAN]);
    Util_Copy((char *)(temp->data), channel, ALLOC_CHAN);
    IRC_Userhost(nick);
  } else {
    Auth_PUser temp;
    temp = Auth_TUser::MatchNick(nick);
    if (temp!=NULL) {
      if (channel!=NULL)
	temp->RemTracking(channel);
      else
	temp->FlushTracking();
    } else {
      Auth_PTicket ticket = Auth_TTicket::MatchNick(nick);
      if (ticket!=NULL) {
	if (channel==NULL)
	  ticket->FlushTracking();
	else
	  if (ticket->IsTracking(channel))
	    ticket->RemTracking(channel);
      }
    }
  }
}

void Auth_TrackCR(Auth_PUHCB uhcb)
{
  Auth_PUser temp = Auth_TUser::MatchUH(uhcb->GetUser(), uhcb->GetHost());
  if (temp!=NULL)
  {
    if (uhcb->data!=NULL)  // This should always be true
      temp->StartTracking((const char *)(uhcb->data), uhcb->GetNick());
    return ;
  }
  new Auth_TTicket(uhcb->GetNick(), uhcb->GetUser(), uhcb->GetHost(),
		   (const char *)(uhcb->data));
  // (This may look slightly confusing; we are allocating a new
  // Auth_TTicket on the fly; the pointer is discarded since the
  // Auth_TTicket class is derived from AutoList so its placement
  // has already been handled by the constructor.
}

void Auth_TrackPr(const char *prefix, const char *channel)
{
  Auth_PUser tuser;
  if ((tuser = Auth_TUser::MatchPrefix(prefix))!=NULL) {
    if (tuser->IsTracking(NULL)) {
      if (!tuser->IsTracking(channel))
	tuser->AddTracking(channel);
    } else
      tuser->StartTracking(channel, IRC_HostmaskNick(prefix));
    return;
  }
  Auth_PTicket temp;
  if ((temp = Auth_TTicket::MatchNick(IRC_HostmaskNick(prefix)))!=NULL) {
    if (!temp->IsTracking(channel))
      temp->AddTracking(channel);
  } else {
    char nick[ALLOC_NICK], user[ALLOC_USER];
    Util_Copy(nick, IRC_HostmaskNick(prefix), ALLOC_NICK);
    Util_Copy(user, IRC_HostmaskUser(prefix), ALLOC_USER);
    new Auth_TTicket(nick, user, IRC_HostmaskHost(prefix), channel);
  }
}

void Auth_RemTrack(const char *channel)
{
  Auth_PUser tuser;
  Auth_PTicket tticket;

  tuser = Auth_TUser::GetFirst();
  while (tuser!=NULL) {
    tuser->RemTracking(channel);
    tuser = tuser->GetNext();
  }
  tticket = Auth_TTicket::GetFirst();
  while (tticket!=NULL) {
    tticket->RemTracking(channel);
    tticket = tticket->GetNext();
  }
}

void Auth_FlushTrack()
{
  Auth_TUser::FlushAllTracking();
  Auth_TTicket::FlushList();
}

void Auth_Rem(void *ticket, uint8 auth)
{
  if (auth<CMD_AUTH_USER)
    return ;
  Auth_PUser temp;
  if (auth==CMD_AUTH_MASTER)
    temp = Auth_TUser::GetMaster();
  else
    temp = (Auth_PUser)(ticket);
  if (temp==NULL) {
    fputs("Error, null ticket on logout or no master!\n", stderr);
    return ;
  }
  IRC_PCL chans;
  if ((chans = temp->GetChans())!=NULL)
    new Auth_TTicket(temp->GetNick(), temp->GetUser(), temp->GetHost(),
		     NULL, chans);
  temp->Logout();
}

void Auth_Userhost(const char *nick, const char *user, const char *host)
{
  Auth_TUHCB::UHCR(nick, user, host);
}

void Auth_TUHCB::UHCR(const char *nick, const char *user, const char *host)
{
  // This is the response to a Userhost challenge.
#ifdef DEBUG
  fputs("DEBUG - Auth_TUHCB::UHCR called\n", stderr);
#endif
  Auth_PUHCB temp;
  for (temp=first;temp!=NULL;temp=temp->GetNext())
    if (strcasecmp(temp->nick, nick)==0) {
      Util_Copy(temp->user, user, ALLOC_USER);
      Util_Copy(temp->host, host, ALLOC_HOST);
      (*(temp->f))(temp);
      Delete(temp);
      return;
    }
}

void Auth_Nick(const char *prefix, const char *nick)
{
  Auth_PUser temp;
  temp = Auth_TUser::MatchPrefix(prefix);
  if (temp!=NULL&&temp->IsTracking(NULL))
    temp->ChangeTracking(nick);
  else {
    Auth_PTicket ticket = Auth_TTicket::MatchNick(IRC_HostmaskNick(prefix));
    if (ticket!=NULL)
      ticket->ChangeTracking(nick);
    else
      fputs("Error: No matching user or tracking ticket on nick change.\n",
	    stderr);
  }
}

void Auth_DCCRequestPfx(const char *prefix, void *cookie)
{
  Auth_PUser temp;
  void *oldcookie;
  temp = Auth_TUser::MatchPrefix(prefix);
  if (temp==NULL)
    IRC_DCCChatReject(cookie);
  else {
    oldcookie = temp->RegisterDCC(cookie);
    IRC_DCCChatAccept(cookie);
    temp->SetDCCStatus(TRUE);
    if (oldcookie!=NULL)
      IRC_DCCChatClose(oldcookie);
  }
}

void Auth_DCCRequest(const char *nick, void *cookie)
{
  // Associate a cookie with a user.
  Auth_PUser temp;
  void *oldcookie;
  temp = Auth_TUser::MatchNick(nick);
  if (temp==NULL) {
    temp = Auth_TUser::MatchNick(nick, TRUE);
    if (temp==NULL)
      IRC_DCCChatReject(cookie);
    else {
      Auth_PUHCB uhcb = new Auth_TUHCB(&Auth_DCCRequestCR, nick);
      uhcb->data = cookie;
      IRC_Userhost(nick);
    }
    return ;
  }
  oldcookie = temp->RegisterDCC(cookie);
  IRC_DCCChatAccept(cookie);
  //  temp->SetDCCStatus(TRUE);
  if (oldcookie!=NULL)
    IRC_DCCChatClose(oldcookie);
}

void Auth_DCCRequestCR(Auth_PUHCB uhcb)
{
  Auth_PUser temp = Auth_TUser::MatchNick(uhcb->GetNick(), TRUE);
  if (temp!=NULL)
    if (strcmp(temp->GetUser(), uhcb->GetUser())==0 &&
	strcmp(temp->GetHost(), uhcb->GetHost())==0) {
      void *oldcookie = temp->RegisterDCC(uhcb->data);
      IRC_DCCChatAccept(uhcb->data);
      //      temp->SetDCCStatus(TRUE);
      if (oldcookie!=NULL)
	IRC_DCCChatClose(oldcookie);
      return ;
    }
  IRC_Notice(uhcb->GetNick(), "You are not an authorized user.");
  IRC_DCCChatReject(uhcb->data);
}

void Auth_DCCEstablished(void *cookie)
{
  Auth_PUser temp;
  temp = Auth_TUser::MatchDCCCookie(cookie, TRUE);
  if (temp==NULL) {  // This shouldn't happen.
    IRC_DCCChatMessage(cookie, "Error: Missing associated account data.");
    IRC_DCCChatClose(cookie);
    return ;
  }
  temp->SetDCCStatus(TRUE);
}

void Auth_DCCLost(void *cookie)
{
  Auth_PUser temp;
  temp = Auth_TUser::MatchDCCCookie(cookie);
  if (temp==NULL) {  // This shouldn't happen
    temp = Auth_TUser::MatchDCCCookie(cookie, TRUE);
    if (temp==NULL) { // This REALLY shouldn't happen
      fputs("Auth_DCCLost: Missing DCC Cookie.\n", stderr);
      return ;
    }
  }
  temp->SetDCCStatus(FALSE);
}

Auth_PUser Auth_TUser::MatchPrefix(const char *prefix)
{
  Auth_PUser temp;
  char user[ALLOC_USER];

  temp = MatchNick(IRC_HostmaskNick(prefix));
  if (temp!=NULL)
    return temp;
  Util_Copy(user, IRC_HostmaskUser(prefix), ALLOC_USER);
  temp = MatchUH(user, IRC_HostmaskHost(prefix));
  if (temp!=NULL&&!temp->security)
    return temp;
  return NULL;
}

Auth_PUser Auth_TUser::MatchNick(const char *nick, TF flag)
{
  Auth_PUser temp;
  temp = first;
  while (temp!=NULL) {
    if (*temp->nick)
      if (temp->chans!=NULL||(flag&&!temp->security))
	if (strcasecmp(temp->nick, nick)==0)
	  return temp;
    temp = temp->GetNext();
  }
  return NULL;
}

Auth_PTicket Auth_TTicket::MatchNick(const char *nick)
{
  CleanStale();
  Auth_PTicket temp;
  temp = first;
  while (temp!=NULL) {
    if (*temp->nick)
      if (strcasecmp(temp->nick, nick)==0)
	return temp;
    temp = temp->GetNext();
  }
  return NULL;
}

Auth_PUser Auth_TUser::MatchUH(const char *user, const char *host)
{
  Auth_PUser temp, res;
  temp = first;
  res = NULL;
  while (temp!=NULL) {
    if (*temp->user&&*temp->host)
      if (!temp->security)
	if (strcasecmp(temp->user, user)==0)
	  if (strcasecmp(temp->host, host)==0) {
	    if (res!=NULL)
	      return NULL; // Ignore multiple matches.
	    res = temp;
	  }
    temp = temp->GetNext();
  }
  return res;
}

Auth_PUser Auth_TUser::MatchDCCCookie(void *cookie, TF flag)
{
  Auth_PUser temp;
  temp = first;
  while (temp!=NULL) {
    if (flag||temp->dcc_active)
      if (temp->dcc_cookie==cookie)
	return temp;
    temp = temp->GetNext();
  }
  return NULL;
}

void Auth_Init()
{
#ifdef DEBUG
  fputs("DEBUG - Auth_Init called\n", stderr);
#endif
  LD_AuthGetUsers();
  Cmd_Init();
}

void Auth_Cleanup()
{
#ifdef DEBUG
  fputs("DEBUG - Auth_Cleanup called\n", stderr);
#endif
  LD_AuthDumpUsers();
  Auth_TTicket::FlushList();
  Auth_TUHCB::FlushList();
}

void Auth_Connect()
{
  // We may do something here eventually.
}

void Auth_Disconnect(const char *message)
{
  Auth_TUser::FlushAllTracking();
}

void Auth_CmdResult(void *ticket, uint8 auth, const char *message)
{
  switch (auth) {
  case CMD_AUTH_NONE:
    ((Auth_PTicket)(ticket))->SendMessage(message);
    return;
  case CMD_AUTH_USER:
    ((Auth_PUser)(ticket))->SendMessage(message);
    return;
  case CMD_AUTH_MASTER:
    Auth_TUser::GetMaster()->SendMessage(message);
    return;
  default:
    fprintf(stderr, "Error: Invalid auth level: %d\n", auth);
  }
}

void Auth_DCCMessage(void *cookie, const char *message)
{
  Auth_PUser temp = Auth_TUser::MatchDCCCookie(cookie);
  if (temp==NULL) {
    fputs("DCC message received from unmatched cookie!", stderr);
    return ;
  }
  if (temp == Auth_TUser::GetMaster())
    Cmd_Root(NULL, CMD_AUTH_MASTER, message);
  else
    Cmd_Root((void *)(temp), CMD_AUTH_USER, message);
}

void Auth_IRCMessage(const char *prefix, const char *message)
{
  Auth_PUser tuser;
  if ((tuser = Auth_TUser::MatchPrefix(prefix))!=NULL) {
    if (tuser == Auth_TUser::GetMaster())
      Cmd_Root(NULL, CMD_AUTH_MASTER, message);
    else
      Cmd_Root((void *)(tuser), CMD_AUTH_USER, message);
    return ;
  }
  Auth_PTicket ticket;
  if ((ticket = Auth_TTicket::MatchNick(IRC_HostmaskNick(prefix)))!=NULL)
    Cmd_Root((void *)(ticket), CMD_AUTH_NONE, message);
  else {
    char nick[ALLOC_NICK], user[ALLOC_USER];
    Util_Copy(nick, IRC_HostmaskNick(prefix), ALLOC_NICK);
    Util_Copy(user, IRC_HostmaskUser(prefix), ALLOC_USER);
    ticket = new Auth_TTicket(nick, user, IRC_HostmaskHost(prefix), 
			      "boguschan");
    Cmd_Root((void *)(ticket), CMD_AUTH_NONE, message);
    if ((ticket = Auth_TTicket::MatchNick(nick))!=NULL)
      ticket->FlushTracking();
  }
}

void Auth_NotifyMaster(const char *message)
{
  Auth_PUser temp = Auth_TUser::GetMaster();
  if (temp==NULL) {
    fputs("Auth_NotifyMaster: No master present in userlist!\n", stderr);
    return;
  }
  temp->SendMessage(message);
}

void Auth_TUHCB::FlushList()
{
  while (first!=NULL)
    Delete(first);
}

Auth_TUHCB::Auth_TUHCB(void (*fun)(Auth_PUHCB), const char *s_nick)
{
#ifdef DEBUG
  fputs("DEBUG - Auth_TUHCB ctor called\n", stderr);
#endif
  if (last!=NULL) {
    prev = last;
    prev->next = this;
    next = NULL;
    last = this;
  } else {
    first = last = this;
    prev = next = NULL;
  }
  f = fun;
  Util_Copy(nick, s_nick, ALLOC_NICK);
  *user = *host = 0;
  data = NULL;
}

void Auth_TUHCB::Delete(Auth_PUHCB uhcb)
{
#ifdef DEBUG
  fputs("DEBUG - Auth_TUHCB::Delete called\n", stderr);
#endif
  if (uhcb->prev!=NULL)
    uhcb->prev->next = uhcb->next;
  if (uhcb->next!=NULL)
    uhcb->next->prev = uhcb->prev;
  if (first==uhcb)
    first = uhcb->next;
  if (last==uhcb)
    last = uhcb->prev;
  delete [] uhcb;
}
