/*
 * Lachesis - Interface to Parsecfg
 * Copyright 2002 M. Dennis
 *
 * This program is free software; please see the file COPYING for details.
 */

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <dirent.h>
#include <stdio.h>
#include "auth.h"
#include "lachesis.h"
#include "alloc.h"
#include "types.h"
#include "utils.h"

extern CmdOpts options;

TF LD_AuthGetUsers()
{
  struct dirent *dirlist, *flglist;
  DIR *dir, *flg;
  FILE *in;
  Auth_PUser tuser;
  char *input = NULL, *masteracct = NULL, *path;
  char acct[ALLOC_ACCT], pass[ALLOC_PASS];
  size_t len = 0;
  bool tbl;
  path = Util_FormatA("%s/users/master", options.datadir);
  in = fopen(path, "r");
  free(path);
  if (in!=NULL) {
    Util_Getline(&masteracct, &len, in);
    *strchr(masteracct, '\n') = '\0';
    fclose(in);
  } else {
    fputs("LoadData: No Master account specified! Failed...\n", stderr);
    return FALSE;
  }
  len = 0;
  path = Util_FormatA("%s/users", options.datadir);
  dir = opendir(path);
  free(path);
  if (dir==NULL) {
    fprintf(stderr, "LoadData: Unexpected error from opendir: %m\n");
    free (masteracct);
    return FALSE;
  }
  while ((dirlist=readdir(dir))!=NULL) {
    if (dirlist->d_type!=DT_DIR || strcmp(dirlist->d_name, ".")==0 || 
	strcmp(dirlist->d_name, "..")==0)
      continue;
    Util_Copy(acct, dirlist->d_name, ALLOC_ACCT);
    path = Util_FormatA("%s/users/%s/password", options.datadir, acct);
    in = fopen(path, "r");
    free(path);
    if (in==NULL) {
      fprintf(stderr, "LoadData: Error retrieving password for %s: %m\n",
	      acct);
      free (masteracct);
      return FALSE;
    }
    Util_Getline(&input, &len, in);
    *strchr(input, '\n') = '\0';
    fclose(in);
    tbl = strcasecmp(acct, masteracct)==0;
    tuser = new Auth_TUser(acct, input, tbl!=FALSE);
    if (tuser!=NULL) {
      TF nosec = FALSE, usemsg = FALSE, nosounds = FALSE, debug = FALSE;
      path = Util_FormatA("%s/users/%s/flags", options.datadir, acct);
      flg = opendir(path);
      free(path);
      while((flglist=readdir(flg))!=NULL) {
	if (flglist->d_type!=DT_REG)
	  continue;
	if (strcmp(flglist->d_name, "nosec")==0) {
	  nosec = TRUE;
	  continue;
	}
	if (strcmp(flglist->d_name, "usemsg")==0) {
	  usemsg = TRUE;
	  continue;
	}
#ifndef RPGSERV
	if (strcmp(flglist->d_name, "nosounds")==0) {
	  nosounds = TRUE;
	  continue;
	}
	if (strcmp(flglist->d_name, "debug")==0)
	  debug = TRUE;
#endif
      }
      tuser->security = !nosec;
      tuser->usemsg = usemsg;
#ifndef RPGSERV
      tuser->sounds = !nosounds;
      tuser->debug = debug;
#endif
    }
  }
  free(masteracct);
  free(input);
  if (Auth_TUser::GetMaster()==NULL) {
    fputs("LoadData: No account is master! Failed...\n", stderr);
    return FALSE;
  }
  return TRUE;
}

void LD_AuthDumpUsers()
{
  // Eventually, this may handle saving the userlist.
  // 20021230 Karinu - No. We perform all changes as made, so there is no
  // reason to save the userlist on exit.
  Auth_TUser::FlushList();
}

void LD_SetSecurity(const char *acct, TF setting)
{
  char *path, *empty;
  path = Util_FormatA("%s/users/%s/flags/nosec", options.datadir, acct);
  if (setting)
    unlink(path);
  else {
    empty = Util_FormatA("%s/empty", options.datadir);
    link(empty, path);
    free(empty);
  }
  free(path);
}

void LD_SetUseMSG(const char *acct, TF setting)
{
  char *path, *empty;
  path = Util_FormatA("%s/users/%s/flags/usemsg", options.datadir, acct);
  if (setting) {
    empty = Util_FormatA("%s/empty", options.datadir);
    link(empty, path);
    free(empty);
  } else
    unlink(path);
  free(path);
}

#ifndef RPGSERV
void LD_SetSounds(const char *acct, TF setting)
{
  char *path, *empty;
  path = Util_FormatA("%s/users/%s/flags/nosounds", options.datadir, acct);
  if (setting)
    unlink(path);
  else {
    empty = Util_FormatA("%s/empty", options.datadir);
    link(empty, path);
    free(empty);
  }
  free(path);
}

#endif
void LD_SetPassword(const char *acct, const char *pass)
{
  // Note: Validity check must occur before this call is made.
  char *path;
  FILE *out;
  int c=0;
  path = Util_FormatA("%s/users/%s/password", options.datadir, acct);
  do {
    c++;
    if (c>10) {
      fprintf(stderr, "File error: unable to write password: %m\n");
      free(path);
      return ;
    }
    unlink(path);
  } while ((out=fopen(path, "wx"))==NULL);
  free(path);
  fprintf(out, "%s\n", pass);
  fclose(out);
}

