/*
 * Lachesis, an IRCRPG combat engine
 * Copyright 2002 M. Dennis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <getopt.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <pwd.h>
#include <signal.h>
#include <time.h>
#include "lachesis.h"
#include "utils.h"
#include "irc.h"
#include "auth.h"
#include "config.h"

static char* id="$(#) $Id: lachesis.cc,v 1.1.1.1.2.2 2004/03/17 05:35:53 lachesis Exp $";

CmdOpts options;

#ifndef NO_USERID
char *username;
#endif

const char *OPTION_SHORT = "s:DPoOLn:N:u:U:Q:c:k:p:l:h::";

const char *OPTION_LIST = 
  "ircnick:Rn:ircname:RN:ircuser:Ru:userinfo:RU:datadir:R1:quitmsg:RQ:\
channel:Rc:chankey:Rk:port:Rp:logfile:Rl:logooc:No:logother:NO:nodesccmt:NL:\
logtitle:R3:help:Oh:pid:NP:pubdice:ND:single:Rs:";

const char *SOFTWARE_NAME =
#ifdef RPGSERV
			    "RPGServ";
#else
			    "Lachesis";
#endif

const char *OTHER_OPTIONS =
#ifdef RPGSERV
"	-s <user>,		Specify a user account who will have sole\n\
	--single=<user>		access to this RPGServ instance. The master\n\
				will always have access.\n";
#else
"";
#endif

#define SHOW_MAINHELP() printf("%s version %s, Copyright %s M. Dennis\n\
%s comes with ABSOLUTELY NO WARRANTY.\n\
This is free software, and you are welcome to redistribute it\n\
under certain conditions; for details see the file COPYING.\n\
\n\
Usage: %s [OPTIONS] [server]\n\
\n\
Available options:\n\
	-h [<opt>],		If no option is specified, display this\n\
	--help[=<opt>]		message. Otherwise, display help on the option.\n\
	-n <nick>,		Specify a nick for the bot to use on IRC.\n\
	--ircnick=<nick>	(limit %d characters)\n\
	-N <name>,		Specify a \"Real Name\" for the bot to use\n\
	--ircname=<name>	on IRC.  (limit %d characters)\n\
	-u <user>,		Specify the userid the bot will give to the\n\
	--ircuser=<user>	IRC server.  (limit %d characters)\n\
	-p <port>,		Specify the port to which to connect.\n\
	--port=<port>		(default 6667)\n\
	-c <channel>,		Specify a channel to join upon connecting.\n\
	--channel=<chan>	If no type specifier is present '#' will be\n\
				assumed. If you want to specify type, remember\n\
				to escape the # or & characters.\n\
				(limit %d characters)\n\
	-k <key>,		Specify a channel key to use on the initial\n\
	--chankey=<key>		join. You only need this if the channel is\n\
				locked with a key (mode +k). If you did not\n\
				specify a channel with -c or --channel this\n\
				key will not be used. (limit %d characters)\n\
\n\
	-U <message>,		Set a USERINFO message. If set, this message is\n\
	--userinfo=<message>	seen by anyone who sends a USERINFO or FINGER\n\
				request to the bot.  (limit %d characters)\n\
	-Q <message>,		Set a quit message. If set, this message is \n\
	--quitmsg=<message>	used as the default signoff message for IRC.\n\
				(limit %d characters)\n\
\n\
	--datadir=<path>	Specify an alternate data directory.\n\
				(default %s)\n\
\n\
	-l <path>,		Specify a file to perform channel logging.\n\
	--logfile=<path>\n\
	-L, --nodesccmt		Skip commenting on non-master descs.\n\
	-o, --logooc		Log OOC comments, not just corrections.\n\
	-O, --logother		Log channels other than the current.\n\
	--logtitle=<title>	Specify title for log.\n\
\n\
	-P, --pid		Drop a pid file\n\
	-D, --pubdice		Use public dice rolls.\n%s",\
SOFTWARE_NAME, Lachesis_Version(), Lachesis_CYear(), SOFTWARE_NAME, argv[0],\
ALLOC_NICK-1, ALLOC_NAME-1, ALLOC_USER-1, ALLOC_CHAN-1, ALLOC_BUFFER-1,\
ALLOC_BUFFER-1, ALLOC_BUFFER-1, options.datadir, OTHER_OPTIONS)

const char *MAIN_HELP_DETAIL[] = {
  "D", "pubdice",
"This will cause the bot to notice the channel instead of the user when\n\
dice are rolled.\n",
  "P", "pid",
"This will cause the bot to drop its pid into a file in the current\n\
directory.\n",
  "l", "logfile",
"This option allows you to specify a file to which the bot will output what\n\
it sees in the channel. The logs are produced in HTML format and customized\n\
for recording IRCRPGs wherein certain text conventions are used. See log.cc\n\
for more details.\n",
  "L", "nodesccmt",
"Disables the otherwise automatic generation of a warning comment when a\n\
description is received from a user other than the bot's master. The comment\n\
will still be generated for a description is received from a non-user.\n",
  "o", "logooc",
"Log OOC comments in the channel as HTML comments. Normally, the only OOC\n\
comments that are logged are those marked as corrections.\n",
  "O", "logother",
"Log data from all channels, not just the current channel. In the future it\n\
may be possible to specify a separate logfile for any given channel.\n",
  "", "logtitle",
"Specify the string to go in the TITLE tags of the log. If not specified a\n\
default is used.\n",
  "n", "ircnick",
"This option allows you to specify the \"nickname\" that the bot can use on\n\
the IRC network. This is what you specify when sending private messages.\n\
You may want to use this option if the default nick is not available on\n\
the IRC network the bot is connecting to.\n",
  "N", "ircname",
"This option allows you to specify the \"Real Name\" that the bot gives to\n\
the IRC server on connect. It is seen in parenthesis on the first line of\n\
a WHOIS and is shown right after hop count on WHO. Most people place some\n\
sort of phrase or message here rather than a name.\n",
  "u", "ircuser",
#ifdef NO_USERID
"This option allows you to specify the username field for the UH mask of\n\
the bot. Note that some servers incorrectly ignore this option in favor of\n\
a response from an identd server if present. If not specified, a default\n\
username is used.\n",
#else
"This option allows you to specify the username field for the UH mask of\n\
the bot. Note that some servers incorrectly ignore this option in favor of\n\
a response from an identd server if present. If not specified, the user\n\
that started the bot is used.\n",
#endif
  "p", "port",
"This option allows you to specify the port of the server to connect to.\n\
Most IRC servers listen on the default port of 6667, but some listen to\n\
multiple ports and/or require bots to connect to a different port for\n\
traffic control purposes. Also, some systems may run multiple IRC servers\n\
on the same IP address, or may have port 6667 blocked; those servers will\n\
need to use a different address.\n\n\
If you have been given a port number for an IRC server other than 6667,\n\
specify it here. Sometimes a server name may be listed as name:number, if\n\
so, you should put the number here and just enter the name as the server\n\
to connect to.\n\n\
Sorry to be long-winded, but not all IRC setups are the same; However, most\n\
people will not need to specify this option.\n",
  "c", "channel",
"This option allows you to specify a channel to join immediately upon\n\
connecting to IRC. If not specified, the bot will connect and wait for\n\
its master to log in and give it a join command.\n\n\
If the channel name specified does not start with #, &, or +, a # will\n\
be added automatically. If you do use # or &, you may need to use a \\\n\
character to prevent confusing the shell, as most shells consider # to mean\n\
that the rest of the line is a comment, and & to mean that everything before\n\
it is a command to be executed in the background, and everything after it is\n\
a separate command. Surrounding the channel name in double-quotes (\"\")\n\
should work as well.\n",
  "k", "chankey",
"This option allows you to specify a channel key for the channel set with\n\
-c or --channel. A channel key is needed to enter channels that have been\n\
set with channel mode +k <key>.\n",
  "U", "userinfo",
"This specifies a message to reply with if the bot receives a CTCP USERINFO\n\
or CTCP FINGER request from another IRC client. Traditionally this has\n\
contained an email address. You can put anything you want here.\n",
  "Q", "quitmsg",
"This specifies a message to use when signing off IRC. This will be used by\n\
DO QUIT if no message is specified to the command, or by the EndIRCInterface\n\
function called when the bot is shut down.\n",
  "", "datadir",
"This specifies an alternate directory to obtain permanent data from. This\n\
will allow for multiple configurations without having to run each from a\n\
different directory.\n",
  "s", "single",
#ifdef RPGSERV
"This specifies a user account to which access will be restricted. Only this\n\
account and the master account will be loaded. You may specify the master\n\
account, in which case only the master account will be loaded.\n",
#else
"This options is not available for Lachesis.\n",
#endif
};

extern "C" void Shutdown()
{
#ifdef DEBUG
  Util_Eputs("DEBUG - Shutdown called\n");
#endif
  free(options.datadir);
  if (options.logfile!=NULL)
    free(options.logfile);
  Auth_Cleanup();
  HLIRC_Cleanup();
  if (options.errlog!=NULL)
    fclose(options.errlog);
  exit (0);
}

void do_cleanup(int sgl)
{
#ifdef DEBUG
  Util_Eputs("DEBUG - do_cleanup called\n");
#endif
  free(options.datadir);
  if (options.logfile!=NULL)
    free(options.logfile);
  Auth_Cleanup();
  HLIRC_Cleanup();
  if (options.errlog!=NULL)
    fclose(options.errlog);
  signal(sgl, SIG_DFL);
  raise(sgl);
}

int main(int argc, char **argv)
{
  int c, d, idx, res;
  struct option *opts;
  const char *ts1, *ts2;
  char **names;
  uint16 nameslen;
  *options.channel = *options.key = *options.userinfo = *options.ircnick =
    *options.ircuser = *options.quitmessage = options.port = 
    *options.logtitle = options.logooc = options.logother = 
    options.nodesccmt = options.pubdice = 0;
  options.datadir = options.logfile = NULL;
#ifdef RPGSERV
  *options.single = 0;
#endif
#ifndef STDERR_LOG_FILE
  options.errlog = NULL;
#else
  options.errlog = fopen(STDERR_LOG_FILE, "wx");
  if (options.errlog==NULL) {
    unlink (STDERR_LOG_FILE "~");
    link (STDERR_LOG_FILE, STDERR_LOG_FILE "~");
    unlink (STDERR_LOG_FILE);
    options.errlog = fopen(STDERR_LOG_FILE, "wx");
  }
#endif
  srand(time(0));
  ts1 = OPTION_LIST;
  for (c=0;(ts2=strchr(ts1, ':'))!=NULL;c++)
    ts1 = ts2 + 1;
  c/=2;
  opts = new struct option[c+1];
  nameslen = c;
  names = new char*[c];
  ts1 = OPTION_LIST;
  for (d=0;d<c;d++) {
    names[d] = strdup(Util_Separate(&ts1, ':'));
    opts[d].name = names[d];
    ts2 = Util_Separate(&ts1, ':');
    switch (ts2[0]) {
    case 'N':
      opts[d].has_arg = no_argument;
      break;
    case 'R':
      opts[d].has_arg = required_argument;
      break;
    case 'O':
      opts[d].has_arg = optional_argument;
      break;
    default:
      Util_Eputs("Error - invalid OPTION_LIST member.\n");
      Util_Eprintf("DEBUG: type %d. count %d. sct %d. name %s. nulres %c.\n", ts2[0], d, c, names[d], (ts2==NULL)?'y':'n');
      exit (255);
    }
    opts[d].flag = NULL;
    opts[d].val = ts2[1];
  }
  opts[c].name = NULL;
  opts[c].has_arg = 0;
  opts[c].flag = NULL;
  opts[c].flag = 0;
  while ((res = getopt_long(argc, argv, OPTION_SHORT, opts, &idx))!=-1) {
    switch (res) {
    case '1':
      if (options.datadir!=NULL)
	free(options.datadir);
      options.datadir = strdup(optarg);
      break;
    case '3':
      Util_Copy(options.logtitle, optarg, ALLOC_BUFFER);
      break;
    case 's':
#ifdef RPGSERV
      Util_Copy(options.single, optarg, ALLOC_ACCT);
#else
      fputs("Invalid argument.\n");
      exit (EXIT_FAILURE);
#endif
      break;
    case 'D':
      options.pubdice = TRUE;
      break;
    case 'P':
      {
	char *fn;
	FILE *fp;
	fn = Util_FormatA("%s.pid", argv[0]);
	unlink(fn);
	fp = fopen(fn, "wx");
	free(fn);
	if (fp==NULL)
	  Util_Eprintf("Unable to write %s.pid\nOur pid is %d\n",
		       argv[0], getpid());
	else {
	  fprintf(fp, "%d\n", getpid());
	  fclose(fp);
	}
      }
      break;
    case 'O':
      options.logother = 1;
      break;
    case 'o':
      options.logooc = 1;
      break;
    case 'L':
      options.nodesccmt = 1;
      break;
    case 'l':
      if (options.logfile!=NULL)
	free(options.logfile);
      options.logfile = strdup(optarg);
      break;
    case 'N':
      Util_Copy(options.ircname, optarg, ALLOC_NAME);
      break;
    case 'Q':
      Util_Copy(options.quitmessage, optarg, ALLOC_BUFFER);
      break;
    case 'U':
      Util_Copy(options.userinfo, optarg, ALLOC_BUFFER);
      break;
    case 'c':
      Util_Copy(options.channel, optarg, ALLOC_CHAN);
      break;
    case 'h':
      if (optarg==NULL)
	SHOW_MAINHELP();
      else
	for (c=0;c<sizeof(MAIN_HELP_DETAIL);c+=3)
	  if (strcasecmp(optarg, MAIN_HELP_DETAIL[c+1])==0 ||
	      MAIN_HELP_DETAIL[c][0] && strcasecmp(optarg, MAIN_HELP_DETAIL[c])==0)
	    {
	      puts(MAIN_HELP_DETAIL[c+2]);
	      break;
	    }
      exit (0);
    case 'k':
      Util_Copy(options.key, optarg, ALLOC_BUFFER);
      break ;
    case 'n':
      Util_Copy(options.ircnick, optarg, ALLOC_NICK);
      break;
    case 'p':
      options.port = strtol(optarg, NULL, 10);
      if (!options.port)
	Util_Eputs("Unable to parse port option; ignoring.\n");
      break;
    case 'u':
      Util_Copy(options.ircuser, optarg, ALLOC_USER);
      break;
    }
  }
  if (optind == argc) {
    Util_Eputs("Error: no server specified.\n");
    exit(1);
  }
  Util_Copy(options.server, argv[optind], ALLOC_HOST);

  if (options.datadir==NULL)
    options.datadir = strdup("data");

#ifndef NO_USERID
  struct passwd *pw = getpwuid(getuid());
  username = strdup(pw->pw_name);
#endif

  for (c=0;c<nameslen;c++)
    free (names[c]);
  delete [] names;

  signal (SIGHUP, SIG_IGN);
  if (signal(SIGINT, do_cleanup)==SIG_IGN)
    signal (SIGINT, SIG_IGN);
  signal (SIGTERM, do_cleanup);

  // Launch the various sections of the program.
  Auth_Init();
  HLIRC_BeginIRCInterface();
  // This should not return.
  Util_Eputs("Error: HLIRC_BeginIRCInterface returned!\n");
  Shutdown();
}
